package mods.immibis.redlogic.wires;


import java.lang.reflect.Field;

import mods.immibis.microblocks.api.IMicroblockCoverSystem;
import mods.immibis.redlogic.IRedstoneEmittingTile;
import mods.immibis.redlogic.RedLogicMod;
import mods.immibis.redlogic.Utils;
import net.minecraft.block.Block;
import net.minecraft.block.BlockRedstoneWire;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.INetworkManager;
import net.minecraft.network.packet.Packet132TileEntityData;
import net.minecraftforge.common.ForgeDirection;

public class RedAlloyTile extends WireTile implements IRedstoneEmittingTile {
	private short MAX_STRENGTH = 255;
	
	private short strength;
	private short strengthFromNonWireBlocks; // this is synced to the client, not used by the server
	
	protected boolean signalStrengthSynced;
	protected boolean connectToBlockBelow;
	
	private boolean isUpdatingStrength, recursiveUpdatePending;
	
	private static boolean dontEmitPower = false;
	
	public RedAlloyTile() {
		super(EnumWires.RED_ALLOY);
	}
	
	private int getStrengthFromSurroundingBlocks() {
		
		if(!worldObj.isRemote)
			strengthFromNonWireBlocks = 0;
		
		int newStrength = 0;
		for(int side = 0; side < 6; side++) {
			if(!isWirePresent(side))
				continue;
			
			if(connectToBlockBelow) {
				ForgeDirection wdir = ForgeDirection.VALID_DIRECTIONS[side];
				int x = xCoord + wdir.offsetX;
				int y = yCoord + wdir.offsetY;
				int z = zCoord + wdir.offsetZ;
				
				int thisStrength = Utils.getPowerStrength(worldObj, x, y, z, wdir.ordinal() ^ 1, false);
				newStrength = Math.max(newStrength, Math.min(thisStrength - 1, MAX_STRENGTH));
				
				if(!worldObj.isRemote)
					strengthFromNonWireBlocks = (short)Math.max(strengthFromNonWireBlocks, Math.min(thisStrength - 1, MAX_STRENGTH));
			}
			
			//dontEmitPower = true;
			
			for(int dir = 0; dir < 6; dir++) {
				if(!connectsInDirection(side, dir))
					continue;
				
				ForgeDirection fdir = ForgeDirection.VALID_DIRECTIONS[dir];
				int x = xCoord + fdir.offsetX, y = yCoord + fdir.offsetY, z = zCoord + fdir.offsetZ;
				
				int odir = dir ^ 1;
				
				if(connectsInDirectionAroundCorner(side, dir)) {
					fdir = ForgeDirection.VALID_DIRECTIONS[side];
					x += fdir.offsetX;
					y += fdir.offsetY;
					z += fdir.offsetZ;
					odir = side ^ 1;
				}
				
				if(worldObj.getBlockId(x, y, z) == RedLogicMod.wire.blockID) {
					WireTile ot = (WireTile)worldObj.getBlockTileEntity(x, y, z);
					IMicroblockCoverSystem omcs = ot.getCoverSystem();
					if(!omcs.isEdgeOpen(side, dir ^ 1))
						continue; // a microblock is blocking this connection in the other tile
				}
				
				int thisStrength = Utils.getPowerStrength(worldObj, x, y, z, odir);
				newStrength = Math.max(newStrength, Math.min(thisStrength - 1, MAX_STRENGTH));
				
				if(!worldObj.isRemote) {
					thisStrength = Utils.getPowerStrength(worldObj, x, y, z, odir, false);
					strengthFromNonWireBlocks = (short)Math.max(strengthFromNonWireBlocks, Math.min(thisStrength - 1, MAX_STRENGTH));
				}
			}
			
			//dontEmitPower = false;
		}
		
		if(worldObj.isRemote)
			newStrength = Math.max(newStrength, strengthFromNonWireBlocks);
		
		return newStrength;
	}
	
	private static boolean isRecursivelyUpdating;
	
	@Override
	void onNeighbourBlockChange() {
		super.onNeighbourBlockChange();
		
		if(isUpdatingStrength) {
			recursiveUpdatePending = true;
			return;
		}
		
		// True if this is the first red alloy tile to update, which received an update from something else,
		// and is now causing a whole bunch of red alloy tiles to change.
		// Only this change will be sent to the client, which will then mirror the processing the server does,
		// to save bandwidth.
		// Note that isRecursivelyUpdating is static, so this is not threadsafe, so we avoid accessing it on the client thread.
		boolean wasFirstServerChange = !worldObj.isRemote && !isRecursivelyUpdating;
		if(wasFirstServerChange)
			isRecursivelyUpdating = true;
		
		int oldStrengthFromNonWireBlocks = strengthFromNonWireBlocks;
		
		isUpdatingStrength = true;
		
		int newStrength;
		int startStrength = strength;
		
		do {
			recursiveUpdatePending = false;
			
			int prevStrength = strength;
			strength = 0;
			newStrength = getStrengthFromSurroundingBlocks();
			
			//if(prevStrength != newStrength)
			//	System.out.println(xCoord+","+yCoord+","+zCoord+" red alloy update pass; "+prevStrength+" -> "+newStrength);
			
			if(newStrength < prevStrength) {
				// this is a huge optimization - it results in a "pulse" of 0 strength being sent down the wire
				// when turning off. if there is another source of power further down the wire, that one will block
				// the pulse and propagate backwards, turning the wires back on with the correct strength in 2 updates.
				notifyConnectedWireNeighbours();
				newStrength = getStrengthFromSurroundingBlocks();
			}
			
			strength = (short)newStrength;
			
			if(strength != prevStrength)
				notifyConnectedWireNeighbours();
			
		} while(recursiveUpdatePending);
		
		isUpdatingStrength = false;
		
		//if(startStrength != newStrength)
		//	System.out.println(xCoord+","+yCoord+","+zCoord+" red alloy update; "+startStrength+" -> "+newStrength);
		
		if(strength != startStrength) {
			if(worldObj.isRemote)
				notifyExtendedNeighbourWiresOnClient();
			else
				notifyExtendedNeighbours();
			
			//System.out.println((worldObj.isRemote ? "client " : wasFirstServerChange ? "was first " : "Not first ") + "change at: " + xCoord+","+yCoord+","+zCoord+", new strength: "+strength);
			
			if(worldObj.isRemote || wasFirstServerChange || strengthFromNonWireBlocks != oldStrengthFromNonWireBlocks)
				worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
			
		} else if(!worldObj.isRemote && oldStrengthFromNonWireBlocks != strengthFromNonWireBlocks)
			worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
		
		if(!worldObj.isRemote)
			isRecursivelyUpdating = !wasFirstServerChange;
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		
		strength = tag.getShort("strength");
	}
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		
		tag.setShort("strength", strength);
	}

	@Override
	public boolean canUpdate() {
		return false;
	}
	
	@Override
	public void onDataPacket(INetworkManager net, Packet132TileEntityData pkt) {
		super.onDataPacket(net, pkt);
		
		strength = pkt.customParam1.getShort("_str");
		strengthFromNonWireBlocks = pkt.customParam1.getShort("_snwb");
		
		// The server will only send an update for the first piece of alloy wire that changed strength.
		// It will not send updates for any other pieces that changed as a result.
		// So we have to simulate that on the client as well.
		notifyExtendedNeighbourWiresOnClient();
	}
	
	@Override
	public Packet132TileEntityData getDescriptionPacket() {
		Packet132TileEntityData p = super.getDescriptionPacket();
		p.customParam1.setShort("_str", strength);
		p.customParam1.setShort("_snwb", strengthFromNonWireBlocks);
		return p;
	}
	
	@Override
	public int getWireColour() {
		return (strength/2 + 127) << 16;
	}
	
	@Override
	protected boolean connects(int x, int y, int z, int wireSide, int direction) {
		Block b = Block.blocksList[worldObj.getBlockId(x, y, z)];
		if(b == null)
			return false;
		if(b.canConnectRedstone(worldObj, x, y, z, direction))
			return true;
		return false;
	}

	@Override
	public short getRedstoneStrength() {
		return dontEmitPower ? 0 : strength;
	}
	
	
	
	private static Field wiresProvidePower = BlockRedstoneWire.class.getDeclaredFields()[0];
	static {
		try {
			wiresProvidePower.setAccessible(true);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
		if(wiresProvidePower.getType() != boolean.class)
			throw new AssertionError("field order changed; fix me");
	}

	public boolean canProvideStrongPowerInDirection(int dir) {
		try {
			return connectToBlockBelow && isWirePresent(dir) && wiresProvidePower.getBoolean(Block.redstoneWire);
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	public boolean canProvideWeakPowerInDirection(int dir) {
		try {
			//if(!isWirePresent(dir) && !connectsInDirection(dir))
			//	return false;
			
			if((getSideMask() & ~(1 << (dir ^ 1))) == 0) // There must be a wire on any side except the opposite side
				return false;
			
			if(!wiresProvidePower.getBoolean(Block.redstoneWire))
				return false;
			
			return true;
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Returns the vanilla redstone strength from 0 to 15.
	 */
	public byte getVanillaRedstoneStrength() {
		return (byte)(getRedstoneStrength() / 17);
	}
}
