/*
 * Skyland
 *
 * Copyright (c) 2014 kegare
 * https://github.com/kegare
 *
 * This mod is distributed under the terms of the Minecraft Mod Public License Japanese Translation, or MMPL_J.
 */

package com.kegare.skyland.handler;

import java.util.Set;

import net.minecraft.client.Minecraft;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.monster.IMob;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.event.ClickEvent;
import net.minecraft.init.Items;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.server.MinecraftServer;
import net.minecraft.util.ChatComponentTranslation;
import net.minecraft.util.ChunkCoordinates;
import net.minecraft.util.EnumChatFormatting;
import net.minecraft.util.IChatComponent;
import net.minecraft.util.MathHelper;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraft.world.storage.WorldInfo;
import net.minecraftforge.client.event.RenderGameOverlayEvent;
import net.minecraftforge.event.entity.living.LivingAttackEvent;
import net.minecraftforge.event.entity.living.LivingEvent.LivingUpdateEvent;
import net.minecraftforge.event.entity.living.LivingFallEvent;
import net.minecraftforge.event.entity.player.PlayerEvent;
import net.minecraftforge.event.world.WorldEvent;

import org.apache.commons.lang3.StringUtils;

import com.google.common.collect.Sets;
import com.kegare.skyland.api.SkylandAPI;
import com.kegare.skyland.core.Config;
import com.kegare.skyland.core.Skyland;
import com.kegare.skyland.item.SkyItems;
import com.kegare.skyland.network.DimSyncMessage;
import com.kegare.skyland.network.FallTeleportMessage;
import com.kegare.skyland.network.PlaySoundMessage;
import com.kegare.skyland.util.SkyUtils;
import com.kegare.skyland.util.Version;
import com.kegare.skyland.util.Version.Status;
import com.kegare.skyland.world.WorldProviderSkyland;

import cpw.mods.fml.client.FMLClientHandler;
import cpw.mods.fml.client.event.ConfigChangedEvent.OnConfigChangedEvent;
import cpw.mods.fml.common.FMLCommonHandler;
import cpw.mods.fml.common.ObfuscationReflectionHelper;
import cpw.mods.fml.common.eventhandler.SubscribeEvent;
import cpw.mods.fml.common.gameevent.PlayerEvent.PlayerChangedDimensionEvent;
import cpw.mods.fml.common.gameevent.PlayerEvent.PlayerLoggedInEvent;
import cpw.mods.fml.common.gameevent.PlayerEvent.PlayerLoggedOutEvent;
import cpw.mods.fml.common.gameevent.PlayerEvent.PlayerRespawnEvent;
import cpw.mods.fml.common.gameevent.TickEvent.Phase;
import cpw.mods.fml.common.gameevent.TickEvent.ServerTickEvent;
import cpw.mods.fml.common.network.FMLNetworkEvent.ClientConnectedToServerEvent;
import cpw.mods.fml.common.network.FMLNetworkEvent.ClientDisconnectionFromServerEvent;
import cpw.mods.fml.common.network.FMLNetworkEvent.ServerConnectionFromClientEvent;
import cpw.mods.fml.relauncher.Side;
import cpw.mods.fml.relauncher.SideOnly;

public class SkyEventHooks
{
	public static final SkyEventHooks instance = new SkyEventHooks();

	public static final Set<String> firstJoinPlayers = Sets.newHashSet();
	public static final ThreadLocal<Set<String>> fallTeleportPlayers = new ThreadLocal()
	{
		@Override
		protected Set<String> initialValue()
		{
			return Sets.newHashSet();
		}
	};

	@SideOnly(Side.CLIENT)
	@SubscribeEvent
	public void onConfigChanged(OnConfigChangedEvent event)
	{
		if (event.modID.equals(Skyland.MODID))
		{
			Config.syncConfig();
		}
	}

	@SideOnly(Side.CLIENT)
	@SubscribeEvent
	public void onRenderGameTextOverlay(RenderGameOverlayEvent.Text event)
	{
		if (SkylandAPI.getWorldType() != null)
		{
			return;
		}

		Minecraft mc = FMLClientHandler.instance().getClient();

		if (SkylandAPI.isEntityInSkyland(mc.thePlayer))
		{
			if (mc.gameSettings.showDebugInfo)
			{
				event.left.add("dim: Skyland");
			}
		}
	}

	@SideOnly(Side.CLIENT)
	@SubscribeEvent
	public void onClientConnected(ClientConnectedToServerEvent event)
	{
		if (Version.getStatus() == Status.PENDING || Version.getStatus() == Status.FAILED)
		{
			Version.versionCheck();
		}
		else if (Version.DEV_DEBUG || Config.versionNotify && Version.isOutdated())
		{
			IChatComponent component = new ChatComponentTranslation("skyland.version.message", EnumChatFormatting.AQUA + "Skyland" + EnumChatFormatting.RESET);
			component.appendText(" : " + EnumChatFormatting.YELLOW + Version.getLatest());
			component.getChatStyle().setChatClickEvent(new ClickEvent(ClickEvent.Action.OPEN_URL, Skyland.metadata.url));

			FMLClientHandler.instance().getClient().ingameGUI.getChatGUI().printChatMessage(component);
		}
	}

	@SideOnly(Side.CLIENT)
	@SubscribeEvent
	public void onClientDisconnected(ClientDisconnectionFromServerEvent event)
	{
		Config.syncConfig();
	}

	@SubscribeEvent
	public void onServerConnected(ServerConnectionFromClientEvent event)
	{
		event.manager.scheduleOutboundPacket(Skyland.network.getPacketFrom(new Config()));
		event.manager.scheduleOutboundPacket(Skyland.network.getPacketFrom(new DimSyncMessage(WorldProviderSkyland.getDimData())));
	}

	@SubscribeEvent
	public void onPlayerLoadFromFile(PlayerEvent.LoadFromFile event)
	{
		for (String str : event.playerDirectory.list())
		{
			if (StringUtils.startsWith(str, event.playerUUID))
			{
				return;
			}
		}

		firstJoinPlayers.add(event.playerUUID);
	}

	@SubscribeEvent
	public void onPlayerLoggedIn(PlayerLoggedInEvent event)
	{
		if (event.player instanceof EntityPlayerMP)
		{
			EntityPlayerMP player = (EntityPlayerMP)event.player;

			if (firstJoinPlayers.contains(player.getUniqueID().toString()))
			{
				WorldServer world = player.getServerForPlayer();
				boolean result = false;

				if (world.getWorldInfo().getTerrainType() == SkylandAPI.getWorldType())
				{
					result = SkyUtils.teleportPlayer(player, player.dimension, 0, 254.0D, 0, player.rotationYaw, player.rotationPitch, true);
				}
				else if (Config.skyborn && !SkylandAPI.isEntityInSkyland(player))
				{
					result = SkyUtils.teleportPlayer(player, SkylandAPI.getDimension());
				}

				if (result)
				{
					player.setSpawnChunk(player.getPlayerCoordinates(), true);

					Skyland.network.sendTo(new PlaySoundMessage("skyland:music.game.skyland"), player);
				}
			}
		}
	}

	@SubscribeEvent
	public void onPlayerLoggedOut(PlayerLoggedOutEvent event)
	{
		EntityPlayer player = event.player;
		String uuid = player.getUniqueID().toString();

		firstJoinPlayers.remove(uuid);
		fallTeleportPlayers.get().remove(uuid);
	}

	@SubscribeEvent
	public void onPlayerRespawn(PlayerRespawnEvent event)
	{
		if (event.player instanceof EntityPlayerMP)
		{
			EntityPlayerMP player = (EntityPlayerMP)event.player;

			if (SkylandAPI.isEntityInSkyland(player))
			{
				int x = MathHelper.floor_double(player.posX);
				int y = MathHelper.floor_double(player.posY);
				int z = MathHelper.floor_double(player.posZ);

				if (player.getServerForPlayer().isAirBlock(x, y - 1, z))
				{
					SkyUtils.teleportPlayer(player, player.dimension);
				}
			}
		}
	}

	@SubscribeEvent
	public void onPlayerChangedDimension(PlayerChangedDimensionEvent event)
	{
		if (event.player instanceof EntityPlayerMP)
		{
			EntityPlayerMP player = (EntityPlayerMP)event.player;

			if (event.toDim == SkylandAPI.getDimension())
			{
				WorldServer world = player.getServerForPlayer();
				NBTTagCompound data = player.getEntityData();
				String key = "Skyland:LastTeleportTime";

				if (!data.hasKey(key) || data.getLong(key) + 18000L < world.getTotalWorldTime())
				{
					Skyland.network.sendTo(new PlaySoundMessage("skyland:music.game.skyland"), player);
				}

				data.setLong(key, world.getTotalWorldTime());
			}
		}
	}

	@SubscribeEvent
	public void onServerTick(ServerTickEvent event)
	{
		if (event.phase != Phase.END)
		{
			return;
		}

		MinecraftServer server = FMLCommonHandler.instance().getMinecraftServerInstance();
		WorldServer world = server.worldServerForDimension(0);

		if (world.getWorldInfo().getTerrainType() == SkylandAPI.getWorldType())
		{
			world.prevRainingStrength = 0.0F;
			world.rainingStrength = 0.0F;
			world.prevThunderingStrength = 0.0F;
			world.thunderingStrength = 0.0F;
		}
	}

	@SubscribeEvent
	public void onLivingUpdate(LivingUpdateEvent event)
	{
		if (SkylandAPI.getWorldType() != null)
		{
			return;
		}

		if (event.entityLiving instanceof EntityPlayerMP)
		{
			EntityPlayerMP player = (EntityPlayerMP)event.entityLiving;
			ItemStack current = player.getCurrentEquippedItem();
			boolean feather = current != null && current.getItem() == Items.feather;

			if (player.dimension == 0 || SkylandAPI.isEntityInSkyland(player))
			{
				if (player.isPlayerSleeping() && feather && (int)ObfuscationReflectionHelper.getPrivateValue(EntityPlayer.class, player, "sleepTimer", "field_71076_b") >= 75)
				{
					player.wakeUpPlayer(false, false, true);

					if (!player.capabilities.isCreativeMode && --current.stackSize <= 0)
					{
						player.inventory.setInventorySlotContents(player.inventory.currentItem, null);
					}

					int dim = player.dimension == 0 ? SkylandAPI.getDimension() : 0;
					ChunkCoordinates coord = player.getBedLocation(dim);
					boolean result;

					if (coord == null)
					{
						result = SkyUtils.teleportPlayer(player, dim);
					}
					else
					{
						result = SkyUtils.teleportPlayer(player, dim, coord.posX + 0.5D, coord.posY, coord.posZ + 0.5D, player.rotationYaw, player.rotationPitch, true);
					}

					if (result && player.mcServer.getConfigurationManager().getCurrentPlayerCount() <= 1)
					{
						WorldServer world = player.getServerForPlayer();

						world.provider.resetRainAndThunder();

						if (!SkylandAPI.isEntityInSkyland(player) && world.getGameRules().getGameRuleBooleanValue("doDaylightCycle"))
						{
							WorldInfo worldInfo = SkyUtils.getWorldInfo(world);
							long i = worldInfo.getWorldTime() + 24000L;

							worldInfo.setWorldTime(i - i % 24000L);
						}
					}

					return;
				}
			}

			if (SkylandAPI.isEntityInSkyland(player))
			{
				if (!player.onGround && player.boundingBox.minY <= -30.0D)
				{
					boolean result = false;

					if (feather)
					{
						ChunkCoordinates coord = player.getBedLocation(0);

						if (coord != null)
						{
							result = SkyUtils.teleportPlayer(player, 0, coord.posX + 0.5D, 254.0D, coord.posZ + 0.5D, player.rotationYaw, player.rotationPitch, false);
						}
					}

					if (!result)
					{
						result = SkyUtils.teleportPlayer(player, 0, player.posX, 254.0D, player.posZ, player.rotationYaw, player.rotationPitch, false);
					}

					if (result)
					{
						SkyUtils.setPlayerLocation(player, player.posX, 300.5D, player.posZ);

						fallTeleportPlayers.get().add(player.getUniqueID().toString());

						Skyland.network.sendTo(new FallTeleportMessage(player), player);
					}
				}
			}
		}
	}

	@SubscribeEvent
	public void onLivingFall(LivingFallEvent event)
	{
		if (event.entityLiving instanceof EntityPlayer && fallTeleportPlayers.get().remove(event.entityLiving.getUniqueID().toString()))
		{
			EntityPlayer player = (EntityPlayer)event.entityLiving;
			ItemStack current = player.getCurrentEquippedItem();

			if (current != null && current.getItem() == Items.feather)
			{
				if (!player.capabilities.isCreativeMode && --current.stackSize <= 0)
				{
					player.inventory.setInventorySlotContents(player.inventory.currentItem, null);
				}

				event.setCanceled(true);
			}
		}
	}

	@SubscribeEvent
	public void onLivingAttack(LivingAttackEvent event)
	{
		EntityLivingBase entity = event.entityLiving;

		if (SkylandAPI.isEntityInSkyland(entity) && entity instanceof IMob && event.source.getDamageType().equalsIgnoreCase("player"))
		{
			if (Config.recordSkyland && entity.getRNG().nextInt(100) == 0)
			{
				entity.dropItem(SkyItems.record_skyland, 1);
			}
		}
	}

	@SubscribeEvent
	public void onWorldUnload(WorldEvent.Unload event)
	{
		if (SkylandAPI.getWorldType() != null)
		{
			return;
		}

		World world = event.world;

		if (!world.isRemote)
		{
			if (world.provider.dimensionId == SkylandAPI.getDimension())
			{
				WorldProviderSkyland.saveDimData();
			}
		}
	}
}