/*
 * Skyland
 *
 * Copyright (c) 2014 kegare
 * https://github.com/kegare
 *
 * This mod is distributed under the terms of the Minecraft Mod Public License Japanese Translation, or MMPL_J.
 */

package com.kegare.skyland.util;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Map;
import java.util.concurrent.ForkJoinPool;

import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.init.Blocks;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.ChunkCoordinates;
import net.minecraft.util.MathHelper;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraft.world.storage.DerivedWorldInfo;
import net.minecraft.world.storage.WorldInfo;
import net.minecraftforge.common.DimensionManager;

import com.google.common.collect.Maps;
import com.kegare.skyland.api.SkylandAPI;
import com.kegare.skyland.core.Skyland;
import com.kegare.skyland.world.TeleporterDummy;

import cpw.mods.fml.common.DummyModContainer;
import cpw.mods.fml.common.FMLCommonHandler;
import cpw.mods.fml.common.Loader;
import cpw.mods.fml.common.ModContainer;
import cpw.mods.fml.common.ObfuscationReflectionHelper;

public class SkyUtils
{
	public static boolean mcpc = FMLCommonHandler.instance().getModName().contains("mcpc");

	private static ForkJoinPool pool;

	public static ForkJoinPool getPool()
	{
		if (pool == null || pool.isShutdown())
		{
			pool = new ForkJoinPool();
		}

		return pool;
	}

	public static ModContainer getModContainer()
	{
		ModContainer mod = Loader.instance().getIndexedModList().get(Skyland.MODID);

		if (mod == null)
		{
			mod = Loader.instance().activeModContainer();

			if (mod == null || mod.getModId() != Skyland.MODID)
			{
				return new DummyModContainer(Skyland.metadata);
			}
		}

		return mod;
	}

	public static boolean archiveDirZip(final File dir, final File dest)
	{
		final Path dirPath = dir.toPath();
		final String parent = dir.getName();
		Map<String, String> env = Maps.newHashMap();
		env.put("create", "true");
		URI uri = dest.toURI();

		try
		{
			uri = new URI("jar:" + uri.getScheme(), uri.getPath(), null);
		}
		catch (Exception e)
		{
			return false;
		}

		try (FileSystem zipfs = FileSystems.newFileSystem(uri, env))
		{
			Files.createDirectory(zipfs.getPath(parent));

			for (File file : dir.listFiles())
			{
				if (file.isDirectory())
				{
					Files.walkFileTree(file.toPath(), new SimpleFileVisitor<Path>()
					{
						@Override
						public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException
						{
							Files.copy(file, zipfs.getPath(parent, dirPath.relativize(file).toString()), StandardCopyOption.REPLACE_EXISTING);

							return FileVisitResult.CONTINUE;
						}

						@Override
						public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException
						{
							Files.createDirectory(zipfs.getPath(parent, dirPath.relativize(dir).toString()));

							return FileVisitResult.CONTINUE;
						}
					});
				}
				else
				{
					Files.copy(file.toPath(), zipfs.getPath(parent, file.getName()), StandardCopyOption.REPLACE_EXISTING);
				}
			}

			return true;
		}
		catch (Exception e)
		{
			e.printStackTrace();

			return false;
		}
	}

	public static int compareWithNull(Object o1, Object o2)
	{
		return (o1 == null ? 1 : 0) - (o2 == null ? 1 : 0);
	}

	public static void setPlayerLocation(EntityPlayerMP player, double posX, double posY, double posZ)
	{
		setPlayerLocation(player, posX, posY, posZ, player.rotationYaw, player.rotationPitch);
	}

	public static void setPlayerLocation(EntityPlayerMP player, double posX, double posY, double posZ, float yaw, float pitch)
	{
		player.mountEntity(null);
		player.playerNetServerHandler.setPlayerLocation(posX, posY, posZ, yaw, pitch);
	}

	public static boolean transferPlayer(EntityPlayerMP player, int dim)
	{
		if (dim != player.dimension)
		{
			if (!DimensionManager.isDimensionRegistered(dim))
			{
				return false;
			}

			player.isDead = false;
			player.forceSpawn = true;
			player.mcServer.getConfigurationManager().transferPlayerToDimension(player, dim, new TeleporterDummy(player.mcServer.worldServerForDimension(dim)));
			player.addExperienceLevel(0);

			return true;
		}

		return false;
	}

	public static boolean teleportPlayer(EntityPlayerMP player, int dim)
	{
		transferPlayer(player, dim);

		WorldServer world = player.getServerForPlayer();
		ChunkCoordinates coord;
		String key = "Skyland:LastTeleport." + dim;
		int x, y, z;

		if (player.getEntityData().hasKey(key))
		{
			NBTTagCompound data = player.getEntityData().getCompoundTag(key);
			x = data.getInteger("PosX");
			y = data.getInteger("PosY");
			z = data.getInteger("PosZ");
			coord = new ChunkCoordinates(x, y, z);
		}
		else if (player.getBedLocation(dim) != null)
		{
			coord = player.getBedLocation(dim);
		}
		else if (world.getWorldInfo().getTerrainType() == SkylandAPI.getWorldType())
		{
			coord = new ChunkCoordinates(0, 64, 0);
		}
		else
		{
			coord = world.getSpawnPoint();
		}

		x = coord.posX;
		y = coord.posY;
		z = coord.posZ;

		if (world.isAirBlock(x, y, z) && world.isAirBlock(x, y + 1, z))
		{
			while (world.isAirBlock(x, y - 1, z))
			{
				--y;
			}

			if (!world.isAirBlock(x, y - 1, z) && !world.getBlock(x, y - 1, z).getMaterial().isLiquid())
			{
				setPlayerLocation(player, x + 0.5D, y + 0.5D, z + 0.5D);

				NBTTagCompound data = player.getEntityData().getCompoundTag(key);

				if (data == null)
				{
					data = new NBTTagCompound();
				}

				data.setInteger("PosX", x);
				data.setInteger("PosY", y);
				data.setInteger("PosZ", z);
				player.getEntityData().setTag(key, data);

				return true;
			}
		}
		else
		{
			int range = 16;

			for (x = coord.posX - range; x < coord.posX + range; ++x)
			{
				for (z = coord.posZ - range; z < coord.posZ + range; ++z)
				{
					for (y = world.getActualHeight() - 3; y > world.provider.getAverageGroundLevel(); --y)
					{
						if (world.isAirBlock(x, y, z) && world.isAirBlock(x, y + 1, z) &&
							world.isAirBlock(x - 1, y, z) && world.isAirBlock(x - 1, y + 1, z) &&
							world.isAirBlock(x + 1, y, z) && world.isAirBlock(x + 1, y + 1, z) &&
							world.isAirBlock(x, y, z - 1) && world.isAirBlock(x, y + 1, z - 1) &&
							world.isAirBlock(x, y, z + 1) && world.isAirBlock(x, y + 1, z + 1))
						{
							while (world.isAirBlock(x, y - 1, z))
							{
								--y;
							}

							if (!world.isAirBlock(x, y - 1, z) && !world.getBlock(x, y - 1, z).getMaterial().isLiquid())
							{
								setPlayerLocation(player, x + 0.5D, y + 0.5D, z + 0.5D);

								NBTTagCompound data = player.getEntityData().getCompoundTag(key);

								if (data == null)
								{
									data = new NBTTagCompound();
								}

								data.setInteger("PosX", x);
								data.setInteger("PosY", y);
								data.setInteger("PosZ", z);
								player.getEntityData().setTag(key, data);

								return true;
							}
						}
					}
				}
			}

			x = 0;
			y = 64;
			z = 0;
			setPlayerLocation(player, x + 0.5D, y + 0.5D, z + 0.5D);
			world.setBlockToAir(x, y, z);
			world.setBlockToAir(x, y + 1, z);
			world.setBlock(x, y - 1, z, Blocks.dirt);
		}

		return false;
	}

	public static boolean teleportPlayer(EntityPlayerMP player, int dim, double posX, double posY, double posZ, float yaw, float pitch, boolean safe)
	{
		transferPlayer(player, dim);

		WorldServer world = player.getServerForPlayer();
		int x = MathHelper.floor_double(posX);
		int y = MathHelper.floor_double(posY);
		int z = MathHelper.floor_double(posZ);

		if (safe)
		{
			if (world.isAirBlock(x, y, z) && world.isAirBlock(x, y + 1, z))
			{
				while (world.isAirBlock(x, y - 1, z))
				{
					--y;
				}

				if (!world.isAirBlock(x, y - 1, z) && !world.getBlock(x, y - 1, z).getMaterial().isLiquid())
				{
					setPlayerLocation(player, posX, y + 0.5D, posZ, yaw, pitch);

					return true;
				}
			}
		}
		else
		{
			setPlayerLocation(player, posX, y + 0.5D, posZ, yaw, pitch);

			return true;
		}

		return teleportPlayer(player, dim);
	}

	public static WorldInfo getWorldInfo(World world)
	{
		WorldInfo worldInfo = world.getWorldInfo();

		if (worldInfo instanceof DerivedWorldInfo)
		{
			worldInfo = ObfuscationReflectionHelper.getPrivateValue(DerivedWorldInfo.class, (DerivedWorldInfo)worldInfo, "theWorldInfo", "field_76115_a");
		}

		return worldInfo;
	}
}